<?php


namespace UI\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class StoreObraRequest extends FormRequest
{
    public function authorize()
    {
        return true;
    }

    public function rules()
    {
        return [
            'titulo' => 'required|string|max:255',
            'numero_certificado' => 'required|string|max:255',
            'data_registo' => 'required|date',
            'provincia_id' => 'required|integer|exists:provincias,id',
            'genero_id' => 'required|integer|exists:generos,id',
            'suporte' => 'required|string|max:255',
            'classificacao_id' => 'required|integer|exists:classificacoes,id',
            'obra_intelectual' => 'required|in:INEDITA,PUBLICADA',
            'descricao' => 'nullable|string|max:255',
            'observacoes' => 'nullable|string|max:255',
            'autores' => 'required|array|min:1',
            'autores.*.autor_id' => 'required|integer|exists:autores,id',
            'autores.*.vinculo_id' => 'required|integer|exists:vinculos,id',
            'autores.*.observacoes' => 'nullable|string|max:255',
            'documentos' => 'nullable|array',
            'documentos.*.file' => 'nullable|file|max:10240',
            'documentos.0.file' => 'nullable|file|max:10240',
            'documentos.*.tipo_documento_id' => 'nullable|integer|exists:tipo_documentos,id',
        ];
    }

    public function prepareForValidation()
    {
        if ($this->has('autores')) {
            $autores = $this->input('autores');
            if (is_string($autores)) {
                $autores = json_decode($autores, true);
            }

            if (is_array($autores) && !isset($autores[0]) && isset($autores['autor_id'])) {
                $autores = [$autores];
            }

            $this->merge(['autores' => $autores]);
        }

        if ($this->has('documentos')) {
            $documentos = $this->input('documentos');

            if (is_string($documentos)) {
                $documentos = json_decode($documentos, true);
            }

            if (is_array($documentos) && !isset($documentos[0]) && isset($documentos['nome'])) {
                $documentos = [$documentos];
            }

            $this->merge(['documentos' => $documentos]);
        }

        if ($this->hasFile('documentos.0.file')) {
            $this->merge(['documentos' => [
                array_merge($this->input('documentos.0', []), [
                    'file' => $this->file('documentos.0.file')
                ])
            ]]);
        }
    }

    public function messages()
    {
        return [
            'titulo.required' => 'O título é obrigatório',
            'numero_certificado.required' => 'O número de certificado é obrigatório',
            'data_registo.required' => 'A data de registro é obrigatória',
            'provincia_id.required' => 'A província é obrigatória',
            'genero_id.required' => 'O género é obrigatório',
            'suporte.required' => 'O suporte é obrigatório',
            'classificacao_id.required' => 'A classificação é obrigatória',
            'obra_intelectual.required' => 'O status da obra intelectual é obrigatório',
            'obra_intelectual.in' => 'O status da obra intelectual deve ser INEDITA ou PUBLICADA',
            'descricao.required' => 'A descrição é obrigatória',
            'observacoes.required' => 'As observações são obrigatórias',
            'autores.required' => 'Os autores são obrigatórios',
            'autores.*.autor_id.required' => 'O autor é obrigatório',
            'autores.*.autor_id.exists' => 'O autor selecionado não é válido',
            'autores.*.vinculo_id.required' => 'O vínculo é obrigatório',
            'autores.*.vinculo_id.exists' => 'O vínculo selecionado não é válido',
             'documentos.required' => 'Os documentos são opcionais',
             'documentos.*.file.required' => 'O arquivo do documento é opcional',
            'documentos.*.file.max' => 'O arquivo do documento não pode exceder 10MB',
            'documentos.*.tipo_documento_id.required' => 'O tipo de documento é opcional',
            'documentos.*.tipo_documento_id.exists' => 'O tipo de documento selecionado não é válido',
            'documentos.*.nome.required' => 'O nome do documento é opcional',
            'documentos.*.nome.max' => 'O nome do documento não pode exceder 255 caracteres',
            'documentos.*.descricao.required' => 'A descrição do documento é opcional',
            'documentos.*.descricao.max' => 'A descrição do documento não pode exceder 255 caracteres',
        ];
    }
}
