<?php

namespace UI\Http\Controllers;

use App\Http\Controllers\Controller;
use Application\DTOs\RequestData;
use Application\DTOs\UpdateRequestData;
use Application\Shared\Search\Paginate;
use Application\Shared\Search\Search;
use Application\Shared\Search\SearchWithPagination;
use Application\UseCases\CreateRequest;
use Application\UseCases\DeleteRequest;
use Application\UseCases\GetRequestById;
use Application\UseCases\UpdateRequest;
use Domain\Repositories\RequestRepositoryInterface;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use UI\Http\Requests\StoreRequestRequest;

/**
 * @OA\Tag(
 *     name="Pedidos",
 *     description="Endpoints para gestão de pedidos de registo de obras intelectuais"
 * )
 */
class RequestController extends Controller
{
    public function __construct(
        private RequestRepositoryInterface $repo,
        private CreateRequest $createRequest,
        private GetRequestById $getRequestById,
        private DeleteRequest  $deleteRequest,
        private UpdateRequest   $updateRequest,

    ) {}

    /**
     * @OA\Get(
     *     path="/api/v1/pedidos",
     *     summary="Listar pedidos",
     *     description="Retorna uma lista de pedidos de registo de obras intelectuais",
     *     tags={"Pedidos"},
     *     @OA\Response(
     *         response=200,
     *         description="Lista de pedidos retornada com sucesso",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="data", type="array", @OA\Items(type="object"))
     *         )
     *     )
     * )
     */
    public function index(Request $request): JsonResponse
    {
        $page = (int) $request->input('page');
        $perPage = (int) $request->input('perPage');
        $sort = $request->input('sort', 'updated_at');
        $order = $request->input('direction', 'desc');

        $search = new Search([
            'orderBy' => 'created_at',
            'orderByDirection' => 'asc',
        ]);

        $paginate = new Paginate(['withPagination' => true, 'page' => $page, 'perPage' => $perPage]);

        $input  = new SearchWithPagination($search, $paginate);
        $request = $this->repo->list($input);
        return response()->json($request->toArray());
    }

    /**
     * @OA\Post(
     *     path="/api/v1/pedidos",
     *     summary="Criar pedido",
     *     description="Cria um novo pedido de registo de obra intelectual",
     *     tags={"Pedidos"},
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"certificate_number", "registration_date", "province", "author_name", "work_title", "genre", "support", "classification"},
     *             @OA\Property(property="certificate_number", type="string", description="Número do certificado"),
     *             @OA\Property(property="registration_date", type="string", format="date", description="Data de registo"),
     *             @OA\Property(property="province", type="string", description="Província"),
     *             @OA\Property(property="author_name", type="string", description="Nome do autor"),
     *             @OA\Property(property="work_title", type="string", description="Título da obra"),
     *             @OA\Property(property="genre", type="string", description="Género"),
     *             @OA\Property(property="support", type="string", description="Suporte"),
     *             @OA\Property(property="classification", type="string", description="Classificação"),
     *             @OA\Property(property="is_published", type="boolean", description="Se está publicada")
     *         )
     *     ),
     *     @OA\Response(
     *         response=201,
     *         description="Pedido criado com sucesso",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="message", type="string", example="Pedido criado com sucesso"),
     *             @OA\Property(property="data", type="object")
     *         )
     *     ),
     *     @OA\Response(
     *         response=422,
     *         description="Dados de validação inválidos",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="message", type="string", example="Dados de validação inválidos"),
     *             @OA\Property(property="errors", type="object")
     *         )
     *     )
     * )
     */
    public function store(StoreRequestRequest $request): JsonResponse
    {
        $validated = $request->validated();

        // Converte a string da data para DateTime
        $validated['registration_date'] = new \DateTime($validated['registration_date']);

        $requestData = new RequestData(
            certificate_number: $validated['certificate_number'],
            registration_date: $validated['registration_date'],
            province: $validated['province'],
            author_name: $validated['author_name'],
            work_title: $validated['work_title'],
            genre: $validated['genre'],
            support: $validated['support'],
            classification: $validated['classification'],
            is_published: $validated['is_published']
        );



        $result = $this->createRequest->execute($requestData);
        print_r($result);

        return response()->json($result, 201);
    }

    public function update(string $id, StoreRequestRequest $requestStore): JsonResponse
    {
        $validated = $requestStore->validated();
        $validated['registration_date'] = new \DateTime($validated['registration_date']);

        $requestData = new UpdateRequestData(
            id: (int) $id,
            certificate_number: $validated['certificate_number'],
            registration_date: $validated['registration_date'],
            province: $validated['province'],
            author_name: $validated['author_name'],
            work_title: $validated['work_title'],
            genre: $validated['genre'],
            support: $validated['support'],
            classification: $validated['classification'],
            is_published: $validated['is_published']
        );

        $result = $this->updateRequest->execute($requestData);
        return response()->json($result, 200);
    }
    /**
     * @OA\Get(
     *     path="/api/v1/pedidos/{id}",
     *     summary="Obter pedido por ID",
     *     description="Retorna os detalhes de um pedido específico",
     *     tags={"Pedidos"},
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="ID do pedido",
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Pedido encontrado",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="data", type="object")
     *         )
     *     ),
     *     @OA\Response(
     *         response=404,
     *         description="Pedido não encontrado",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="message", type="string", example="Pedido não encontrado")
     *         )
     *     )
     * )
     */
    public function show(string $id): JsonResponse
    {
        $requestId = (int) $id;
        $request = $this->getRequestById->execute($requestId);
        return response()->json($request, 200);
    }
    /**
     * @OA\Delete(
     *     path="/api/v1/pedidos/{id}",
     *     summary="Eliminar pedido",
     *     description="Elimina um pedido de registo de obra intelectual",
     *     tags={"Pedidos"},
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="ID do pedido",
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Pedido eliminado com sucesso",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="message", type="string", example="Pedido eliminado com sucesso")
     *         )
     *     ),
     *     @OA\Response(
     *         response=404,
     *         description="Pedido não encontrado",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="message", type="string", example="Pedido não encontrado")
     *         )
     *     )
     * )
     */
    public function destroy(string $id): JsonResponse
    {
        $requestId = (int) $id;

        $requestDeleted = $this->deleteRequest->execute($requestId);


        return response()->json(['message' => 'Pedido eliminado'], 200);
    }
}
