<?php

namespace UI\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Application\UseCases\CreateObra;
use Application\UseCases\AddDocumentoObra;
use Application\UseCases\AddAutorObra;
use Application\UseCases\ManageObraPublicacao;
use Application\UseCases\RemoveDocumentoObra;
use Application\UseCases\RemoveAutorObra;
use Application\UseCases\UpdateObra;
use Application\DTOs\CreateObraDTO;
use Application\DTOs\UpdateObraDTO;
use Application\DTOs\AddDocumentoObraDTO;
use Application\DTOs\AddAutorObraDTO;
use Application\DTOs\ObraPublicacaoDTO;
use Domain\Repositories\IObraRepository;
use Domain\Repositories\IAutorObraRepository;
use Domain\Repositories\IVinculoRepositoryInterface;
use Domain\Services\IFileStorageService;
use Infrastructure\Persistence\Eloquent\Models\ObraModel;

/**
 * @OA\Tag(
 *     name="Obras",
 *     description="API para gestão de obras intelectuais"
 * )
 */
class ObraController extends Controller
{
    public function __construct(
        private CreateObra $createObraUseCase,
        private UpdateObra $updateObraUseCase,
        private AddDocumentoObra $addDocumentoObraUseCase,
        private AddAutorObra $addAutorObraUseCase,
        private ManageObraPublicacao $manageObraPublicacaoUseCase,
        private RemoveDocumentoObra $removeDocumentoObraUseCase,
        private RemoveAutorObra $removeAutorObraUseCase,
        private IObraRepository $obraRepository,
        private IAutorObraRepository $autorObraRepository,
        private IVinculoRepositoryInterface $vinculoRepository,
        private IFileStorageService $fileStorageService
    ) {}

    /**
     * @OA\Post(
     *     path="/api/v1/obras",
     *     summary="Criar uma nova obra",
     *     description="Cria uma nova obra intelectual com autores e documentos associados.",
     *     tags={"Obras"},
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\MediaType(
     *             mediaType="multipart/form-data",
     *             @OA\Schema(
     *                 required={"titulo", "numero_certificado", "data_registo", "provincia_id", "genero_id", "suporte", "classificacao_id", "obra_intelectual", "autores"},
     *                 @OA\Property(property="titulo", type="string", maxLength=255, description="Título da obra", example="O Meu Livro"),
     *                 @OA\Property(property="numero_certificado", type="string", maxLength=255, description="Número do certificado", example="CERT-2024-001"),
     *                 @OA\Property(property="data_registo", type="string", format="date", description="Data de registo (YYYY-MM-DD)", example="2024-01-15"),
     *                 @OA\Property(property="provincia_id", type="integer", description="ID da província", example=1),
     *                 @OA\Property(property="genero_id", type="integer", description="ID do género", example=1),
     *                 @OA\Property(property="suporte", type="string", maxLength=255, description="Suporte da obra"),
     *                 @OA\Property(property="classificacao_id", type="integer", description="ID da classificação", example=1),
     *                 @OA\Property(property="obra_intelectual", type="string", description="Status da obra intelectual: INEDITA ou PUBLICADA", example="INEDITA"),
     *                 @OA\Property(property="descricao", type="string", description="Descrição da obra", example="Uma obra literária"),
     *                 @OA\Property(property="observacoes", type="string", description="Observações", example="Primeira edição"),
     *                 @OA\Property(property="editora", type="string", description="Nome da editora (obrigatório se obra_intelectual=PUBLICADA)", example="Editora Nacional"),
     *                 @OA\Property(property="num_edicao", type="string", description="Número da edição (obrigatório se obra_intelectual=PUBLICADA)", example="1ª Edição"),
     *                 @OA\Property(property="ano", type="integer", description="Ano de publicação (obrigatório se obra_intelectual=PUBLICADA)", example=2024),
     *                 @OA\Property(property="produtora", type="string", description="Nome da produtora (obrigatório se obra_intelectual=PUBLICADA)", example="Produtora ABC"),
     *                 @OA\Property(property="local_publicacao", type="string", description="Local de publicação (obrigatório se obra_intelectual=PUBLICADA)", example="Luanda, Angola"),
     *                 @OA\Property(property="volume", type="string", description="Volume da obra (obrigatório se obra_intelectual=PUBLICADA)", example="Volume 1"),
     *                 @OA\Property(
     *                     property="autores",
     *                     type="string",
     *                     description="Array JSON de autores com vínculos",
     *                     example="[{'autor_id': 1, 'vinculo_id': 1, 'observacoes': 'Autor principal'}]"
     *                 ),
     *                 @OA\Property(
     *                     property="documentos[0][tipo_documento_id]",
     *                     type="integer",
     *                     description="ID do tipo de documento (1-8)",
     *                     example=1
     *                 ),
     *                 @OA\Property(
     *                     property="documentos[0][file]",
     *                     type="string",
     *                     format="binary",
     *                     description="Ficheiro do documento - clique para selecionar arquivo (máximo 10MB)"
     *                 ),
     *                 @OA\Property(
     *                     property="documentos[1][tipo_documento_id]",
     *                     type="integer",
     *                     description="ID do tipo de documento (1-8)",
     *                     example=2
     *                 ),
     *                 @OA\Property(
     *                     property="documentos[1][file]",
     *                     type="string",
     *                     format="binary",
     *                     description="Ficheiro do documento - clique para selecionar arquivo (máximo 10MB)"
     *                 )
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=201,
     *         description="Obra criada com sucesso",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="message", type="string", example="Obra criada com sucesso"),
     *             @OA\Property(
     *                 property="data",
     *                 type="object",
     *                 @OA\Property(property="id", type="integer", example=1),
     *                 @OA\Property(property="titulo", type="string", example="O Meu Livro"),
     *                 @OA\Property(property="numero_certificado", type="string", example="CERT-2024-001"),
     *                 @OA\Property(property="data_registo", type="string", format="date", example="2024-01-15"),
     *                 @OA\Property(property="provincia_id", type="integer", example=1),
     *                 @OA\Property(property="genero_id", type="integer", example=1),
     *                 @OA\Property(property="suporte", type="string", example="Papel"),
     *                 @OA\Property(property="classificacao_id", type="integer", example=1),
     *                 @OA\Property(property="obra_intelectual", type="string", example="INEDITA"),
     *                 @OA\Property(property="descricao", type="string", example="Uma obra literária"),
     *                 @OA\Property(property="observacoes", type="string", example="Primeira edição"),
     *                 @OA\Property(property="created_at", type="string", format="date-time"),
     *                 @OA\Property(property="updated_at", type="string", format="date-time")
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=422,
     *         description="Dados de validação inválidos",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="message", type="string", example="Dados de validação inválidos"),
     *             @OA\Property(property="errors", type="object")
     *         )
     *     ),
     *     @OA\Response(
     *         response=500,
     *         description="Erro interno do servidor",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="message", type="string", example="Erro interno do servidor")
     *         )
     *     )
     * )
     */
    public function store(Request $request): JsonResponse
    {
        try {

            $data = $request->all();

            $obra_intelectual = 'INEDITA';
            if (isset($data['obra_intelectual'])) {
                $obra_intelectual = $data['obra_intelectual'];
            }

            $autores = [];
            if ($request->has('autores')) {
                $autoresInput = $request->input('autores');

                if (is_string($autoresInput)) {
                    $autoresInput = str_replace("'", '"', $autoresInput);
                    $autoresInput = json_decode($autoresInput, true);

                    if (json_last_error() !== JSON_ERROR_NONE) {
                        throw new \InvalidArgumentException('Formato JSON dos autores inválido: ' . json_last_error_msg());
                    }
                }

                if (!is_array($autoresInput)) {
                    throw new \InvalidArgumentException('Autores devem ser um array válido');
                }

                foreach ($autoresInput as $autorData) {
                    if (is_array($autorData) && isset($autorData['autor_id']) && isset($autorData['vinculo_id'])) {
                        $autores[] = [
                            'autor_id' => (int) $autorData['autor_id'],
                            'vinculo_id' => (int) $autorData['vinculo_id'],
                            'observacoes' => $autorData['observacoes'] ?? null
                        ];
                    }
                }
            }

            if (empty($autores)) {
                throw new \InvalidArgumentException('Autores devem ser um array JSON válido com pelo menos um autor');
            }

            $documentos = $this->processDocumentos($request);

            $validator = Validator::make($data, [
                'titulo' => 'required|string|max:255',
                'numero_certificado' => 'required|string|max:255',
                'data_registo' => 'required|date|before_or_equal:today',
                'provincia_id' => 'required|integer|exists:provincias,id',
                'genero_id' => 'required|integer|exists:generos,id',
                'suporte' => 'required|string|max:255',
                'classificacao_id' => 'required|integer|exists:classificacoes,id',
                'obra_intelectual' => 'required|string|in:INEDITA,PUBLICADA',
                'autores' => 'required|string',
                'descricao' => 'nullable|string',
                'observacoes' => 'nullable|string',
                'editora' => 'nullable|string|max:255',
                'num_edicao' => 'nullable|string|max:50',
                'ano' => 'nullable|integer|min:1900|max:' . date('Y'),
                'produtora' => 'nullable|string|max:255',
                'local_publicacao' => 'nullable|string|max:255',
                'volume' => 'nullable|string|max:50'
            ]);

            if (empty($autores)) {
                $validator->errors()->add('autores', 'Pelo menos um autor deve ser fornecido.');
            } else {
                foreach ($autores as $index => $autor) {
                    if (!isset($autor['autor_id']) || !isset($autor['vinculo_id'])) {
                        $validator->errors()->add("autores.{$index}", 'Autor ID e Vínculo ID são obrigatórios.');
                    }
                }
            }

            if ($obra_intelectual === 'PUBLICADA') {
                if (empty($request->input('editora'))) {
                    $validator->errors()->add('editora', 'Editora é obrigatória para obras PUBLICADAS.');
                }
                if (empty($request->input('ano'))) {
                    $validator->errors()->add('ano', 'Ano de publicação é obrigatório para obras PUBLICADAS.');
                }
            }

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $dto = new CreateObraDTO(
                $request->input('titulo'),
                $request->input('numero_certificado'),
                $request->input('data_registo'),
                $request->input('provincia_id'),
                $request->input('genero_id'),
                $request->input('suporte'),
                $request->input('classificacao_id'),
                $obra_intelectual,
                $autores,
                $documentos,
                $request->input('descricao'),
                $request->input('observacoes'),
                $request->input('editora'),
                $request->input('num_edicao'),
                $request->input('ano'),
                $request->input('produtora'),
                $request->input('local_publicacao'),
                $request->input('volume')
            );

            $obra = $this->createObraUseCase->execute($dto);

            return response()->json([
                'success' => true,
                'message' => 'Obra criada com sucesso',
                'data' => $obra
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Dados de validação inválidos',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro interno do servidor: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * @OA\Put(
     *     path="/api/v1/obras/{id}",
     *     summary="Actualizar uma obra existente",
     *     description="Actualiza uma obra intelectual existente com novos dados.",
     *     tags={"Obras"},
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="ID da obra",
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\MediaType(
     *             mediaType="multipart/form-data",
     *             @OA\Schema(
     *                 required={"titulo", "numero_certificado", "data_registo", "provincia_id", "genero_id", "suporte", "classificacao_id", "obra_intelectual", "autores"},
     *                 @OA\Property(property="titulo", type="string", maxLength=255, description="Título da obra", example="Minha Obra Literária"),
     *                 @OA\Property(property="numero_certificado", type="string", maxLength=50, description="Número do certificado", example="CERT-2024-001"),
     *                 @OA\Property(property="data_registo", type="string", format="date", description="Data de registo", example="2024-01-15"),
     *                 @OA\Property(property="provincia_id", type="integer", description="ID da província", example=1),
     *                 @OA\Property(property="genero_id", type="integer", description="ID do género", example=1),
     *                 @OA\Property(property="suporte", type="string", maxLength=255, description="Suporte da obra"),
     *                 @OA\Property(property="classificacao_id", type="integer", description="ID da classificação", example=1),
     *                 @OA\Property(property="obra_intelectual", type="string", description="Status da obra intelectual: INEDITA ou PUBLICADA", example="INEDITA"),
     *                 @OA\Property(property="descricao", type="string", description="Descrição da obra", example="Uma obra literária"),
     *                 @OA\Property(property="observacoes", type="string", description="Observações", example="Primeira edição"),
     *                 @OA\Property(property="editora", type="string", description="Nome da editora (obrigatório se obra_intelectual=PUBLICADA)", example="Editora Nacional"),
     *                 @OA\Property(property="num_edicao", type="string", description="Número da edição (obrigatório se obra_intelectual=PUBLICADA)", example="1ª Edição"),
     *                 @OA\Property(property="ano", type="integer", description="Ano de publicação (obrigatório se obra_intelectual=PUBLICADA)", example=2024),
     *                 @OA\Property(property="produtora", type="string", description="Nome da produtora (obrigatório se obra_intelectual=PUBLICADA)", example="Produtora ABC"),
     *                 @OA\Property(property="local_publicacao", type="string", description="Local de publicação (obrigatório se obra_intelectual=PUBLICADA)", example="Luanda, Angola"),
     *                 @OA\Property(property="volume", type="string", description="Volume da obra (obrigatório se obra_intelectual=PUBLICADA)", example="Volume 1"),
     *                 @OA\Property(
     *                     property="autores",
     *                     type="string",
     *                     description="Array JSON de autores com vínculos",
     *                     example="[{'autor_id': 1, 'vinculo_id': 1, 'observacoes': 'Autor principal'}]"
     *                 ),
     *                 @OA\Property(
     *                     property="documentos[0][tipo_documento_id]",
     *                     type="integer",
     *                     description="ID do tipo de documento (1-8)",
     *                     example=1
     *                 ),
     *                 @OA\Property(
     *                     property="documentos[0][file]",
     *                     type="string",
     *                     format="binary",
     *                     description="Ficheiro do documento - clique para selecionar arquivo (máximo 10MB)"
     *                 ),
     *                 @OA\Property(
     *                     property="documentos[1][tipo_documento_id]",
     *                     type="integer",
     *                     description="ID do tipo de documento (1-8)",
     *                     example=2
     *                 ),
     *                 @OA\Property(
     *                     property="documentos[1][file]",
     *                     type="string",
     *                     format="binary",
     *                     description="Ficheiro do documento - clique para selecionar arquivo (máximo 10MB)"
     *                 )
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Obra atualizada com sucesso",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="message", type="string", example="Obra atualizada com sucesso"),
     *             @OA\Property(
     *                 property="data",
     *                 type="object",
     *                 @OA\Property(property="id", type="integer", example=1),
     *                 @OA\Property(property="titulo", type="string", example="Minha Obra Literária"),
     *                 @OA\Property(property="numero_certificado", type="string", example="CERT-2024-001"),
     *                 @OA\Property(property="data_registo", type="string", example="2024-01-15"),
     *                 @OA\Property(property="provincia_id", type="integer", example=1),
     *                 @OA\Property(property="genero_id", type="integer", example=1),
     *                 @OA\Property(property="suporte", type="string", example="Digital"),
     *                 @OA\Property(property="classificacao_id", type="integer", example=1),
     *                 @OA\Property(property="obra_intelectual", type="string", example="INEDITA"),
     *                 @OA\Property(property="descricao", type="string", example="Uma obra literária"),
     *                 @OA\Property(property="observacoes", type="string", example="Primeira edição"),
     *                 @OA\Property(property="created_at", type="string", example="2024-01-15T10:30:00Z"),
     *                 @OA\Property(property="updated_at", type="string", example="2024-01-15T10:30:00Z"),
     *                 @OA\Property(
     *                     property="documentos_existentes",
     *                     type="array",
     *                     description="Lista de documentos existentes da obra",
     *                     @OA\Items(
     *                         @OA\Property(property="id", type="integer", example=1),
     *                         @OA\Property(property="nome", type="string", example="documento.pdf"),
     *                         @OA\Property(property="caminho_arquivo", type="string", example="obras/1/documento.pdf"),
     *                         @OA\Property(property="tamanho", type="string", example="1024"),
     *                         @OA\Property(property="extensao", type="string", example="pdf"),
     *                         @OA\Property(property="tipo_documento_id", type="integer", example=1),
     *                         @OA\Property(property="tipo_documento_nome", type="string", example="PDF"),
     *                         @OA\Property(property="ordem", type="integer", example=1),
     *                         @OA\Property(property="created_at", type="string", example="2024-01-15T10:30:00Z"),
     *                         @OA\Property(property="url_download", type="string", example="http://localhost/storage/obras/1/documento.pdf")
     *                     )
     *                 )
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=404,
     *         description="Obra não encontrada",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="message", type="string", example="Obra não encontrada")
     *         )
     *     ),
     *     @OA\Response(
     *         response=422,
     *         description="Dados de validação inválidos",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="message", type="string", example="Dados de validação inválidos"),
     *             @OA\Property(
     *                 property="errors",
     *                 type="object",
     *                 @OA\Property(property="titulo", type="array", @OA\Items(type="string", example="O campo título é obrigatório"))
     *             )
     *         )
     *     )
     * )
     */
    public function update(Request $request, int $id): JsonResponse
    {
        try {
            $obra_intelectual = $request->input('obra_intelectual');

            $autoresJson = $request->input('autores');
            $autores = [];

            if (!empty($autoresJson)) {
                if (is_string($autoresJson)) {
                    $autoresJson = str_replace("'", '"', $autoresJson);
                    $autores = json_decode($autoresJson, true);

                    if (json_last_error() !== JSON_ERROR_NONE) {
                        throw new \InvalidArgumentException('Formato JSON dos autores inválido: ' . json_last_error_msg());
                    }
                } elseif (is_array($autoresJson)) {
                    $autores = $autoresJson;
                }
            }

            if (empty($autores)) {
                $index = 0;
                while ($request->has("autores[{$index}][autor_id]")) {
                    $autores[] = [
                        'autor_id' => (int) $request->input("autores[{$index}][autor_id]"),
                        'vinculo_id' => (int) $request->input("autores[{$index}][vinculo_id]"),
                        'observacoes' => $request->input("autores[{$index}][observacoes]")
                    ];
                    $index++;
                }
            }

            if (empty($autores) || !is_array($autores)) {
                throw new \InvalidArgumentException('Autores devem ser fornecidos em formato válido');
            }

            $documentos = $this->processDocumentos($request);

            $validator = Validator::make($request->all(), [
                'titulo' => 'required|string|max:255',
                'numero_certificado' => 'required|string|max:50',
                'data_registo' => 'required|date',
                'provincia_id' => 'required|integer|exists:provincias,id',
                'genero_id' => 'required|integer|exists:generos,id',
                'suporte' => 'required|string|max:255',
                'classificacao_id' => 'required|integer|exists:classificacoes,id',
                'obra_intelectual' => 'required|in:INEDITA,PUBLICADA',
                'descricao' => 'nullable|string',
                'observacoes' => 'nullable|string',
                'editora' => 'nullable|string|max:255',
                'num_edicao' => 'nullable|string|max:50',
                'ano' => 'nullable|integer|min:1900|max:' . date('Y'),
                'produtora' => 'nullable|string|max:255',
                'local_publicacao' => 'nullable|string|max:255',
                'volume' => 'nullable|string|max:50'
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            if ($obra_intelectual === 'PUBLICADA') {
                if (empty($request->input('editora'))) {
                    throw new \InvalidArgumentException('Editora é obrigatória quando obra_intelectual=PUBLICADA');
                }
                if (empty($request->input('ano'))) {
                    throw new \InvalidArgumentException('Ano é obrigatório quando obra_intelectual=PUBLICADA');
                }
            }

            $dto = new UpdateObraDTO(
                $id,
                $request->input('titulo'),
                $request->input('numero_certificado'),
                $request->input('data_registo'),
                $request->input('provincia_id'),
                $request->input('genero_id'),
                $request->input('suporte'),
                $request->input('classificacao_id'),
                $obra_intelectual,
                $autores,
                $documentos,
                $request->input('descricao'),
                $request->input('observacoes'),
                $request->input('editora'),
                $request->input('num_edicao'),
                $request->input('ano'),
                $request->input('produtora'),
                $request->input('local_publicacao'),
                $request->input('volume')
            );

            $result = $this->updateObraUseCase->execute($dto);

            return response()->json([
                'success' => true,
                'message' => 'Obra atualizada com sucesso',
                'data' => [
                    'id' => $result['obra']->id,
                    'titulo' => $result['obra']->titulo,
                    'numero_certificado' => $result['obra']->numero_certificado,
                    'data_registo' => $result['obra']->data_registo->format('Y-m-d'),
                    'provincia_id' => $result['obra']->provincia_id,
                    'genero_id' => $result['obra']->genero_id,
                    'suporte' => $result['obra']->suporte,
                    'classificacao_id' => $result['obra']->classificacao_id,
                    'obra_intelectual' => $result['obra']->obra_intelectual,
                    'descricao' => $result['obra']->descricao,
                    'observacoes' => $result['obra']->observacoes,
                    'created_at' => $result['obra']->createdAt->format('Y-m-d\TH:i:s\Z'),
                    'updated_at' => $result['obra']->updatedAt->format('Y-m-d\TH:i:s\Z'),
                    'documentos_existentes' => $result['documentos_existentes']
                ]
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Dados de validação inválidos',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao atualizar obra: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * @OA\Get(
     *     path="/api/v1/obras",
     *     summary="Listar todas as obras",
     *     description="Retorna uma lista paginada de todas as obras registadas no sistema com filtros e ordenação",
     *     tags={"Obras"},
     *     @OA\Parameter(
     *         name="page",
     *         in="query",
     *         required=false,
     *         description="Número da página para paginação",
     *         @OA\Schema(type="integer", default=1)
     *     ),
     *     @OA\Parameter(
     *         name="perPage",
     *         in="query",
     *         required=false,
     *         description="Número de itens por página",
     *         @OA\Schema(type="integer", default=20)
     *     ),
     *     @OA\Parameter(
     *         name="sort",
     *         in="query",
     *         required=false,
     *         description="Campo para ordenação",
     *         @OA\Schema(type="string", default="updated_at")
     *     ),
     *     @OA\Parameter(
     *         name="direction",
     *         in="query",
     *         required=false,
     *         description="Direcção da ordenação (asc ou desc)",
     *         @OA\Schema(type="string", default="desc")
     *     ),
     *     @OA\Parameter(
     *         name="titulo",
     *         in="query",
     *         required=false,
     *         description="Filtrar por título da obra (busca parcial)",
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\Parameter(
     *         name="numero_certificado",
     *         in="query",
     *         required=false,
     *         description="Filtrar por número do certificado (busca parcial)",
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\Parameter(
     *         name="obra_intelectual",
     *         in="query",
     *         required=false,
     *         description="Filtrar por status da obra intelectual",
     *         @OA\Schema(type="string", enum={"INEDITA", "PUBLICADA"})
     *     ),
     *     @OA\Parameter(
     *         name="provincia_id",
     *         in="query",
     *         required=false,
     *         description="Filtrar por ID da província",
     *         @OA\Schema(type="integer")
     *     ),
     *     @OA\Parameter(
     *         name="genero_id",
     *         in="query",
     *         required=false,
     *         description="Filtrar por ID do género",
     *         @OA\Schema(type="integer")
     *     ),
     *     @OA\Parameter(
     *         name="classificacao_id",
     *         in="query",
     *         required=false,
     *         description="Filtrar por ID da classificação",
     *         @OA\Schema(type="integer")
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Lista de obras retornada com sucesso",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(
     *                 property="data",
     *                 type="array",
     *                 @OA\Items(
     *                     @OA\Property(property="id", type="integer", example=1),
     *                     @OA\Property(property="titulo", type="string", example="O Meu Livro"),
     *                     @OA\Property(property="numero_certificado", type="string", example="CERT-2024-001"),
     *                     @OA\Property(property="data_registo", type="string", format="date", example="2024-01-15"),
     *                     @OA\Property(property="provincia", type="string", example="Luanda"),
     *                     @OA\Property(property="genero_id", type="integer", example=1),
     *                     @OA\Property(property="suporte", type="string", example="Papel"),
     *                     @OA\Property(property="classificacao_id", type="integer", example=1),
     *                     @OA\Property(property="esta_publicada", type="boolean", example=false),
     *                     @OA\Property(property="descricao", type="string", example="Uma obra literária"),
     *                     @OA\Property(property="observacoes", type="string", example="Primeira edição"),
     *                     @OA\Property(property="created_at", type="string", format="date-time"),
     *                     @OA\Property(property="updated_at", type="string", format="date-time")
     *                 )
     *             ),
     *             @OA\Property(
     *                 property="pagination",
     *                 type="object",
     *                 @OA\Property(property="current_page", type="integer", example=1),
     *                 @OA\Property(property="per_page", type="integer", example=20),
     *                 @OA\Property(property="total", type="integer", example=100),
     *                 @OA\Property(property="last_page", type="integer", example=5)
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=500,
     *         description="Erro interno do servidor",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="message", type="string", example="Erro ao listar obras")
     *         )
     *     )
     * )
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $page = (int) $request->input('page', 1);
            $perPage = (int) $request->input('perPage', 20);
            $sort = $request->input('sort', 'updated_at');
            $order = $request->input('direction', 'desc');

            $allObras = $this->obraRepository->findAll();
            $filteredObras = $this->applyFilters($allObras, $request);
            $sortedObras = $this->applySorting($filteredObras, $sort, $order);
            $total = count($sortedObras);
            $offset = ($page - 1) * $perPage;
            $paginatedObras = array_slice($sortedObras, $offset, $perPage);

            return response()->json([
                'success' => true,
                'data' => $paginatedObras,
                'pagination' => [
                    'current_page' => $page,
                    'per_page' => $perPage,
                    'total' => $total,
                    'last_page' => ceil($total / $perPage)
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    private function applyFilters(array $obras, Request $request): array
    {
        $filters = [
            'titulo' => $request->input('titulo'),
            'numero_certificado' => $request->input('numero_certificado'),
            'obra_intelectual' => $request->input('obra_intelectual'),
            'provincia_id' => $request->input('provincia_id'),
            'genero_id' => $request->input('genero_id'),
            'classificacao_id' => $request->input('classificacao_id'),
        ];

        return array_filter($obras, function ($obra) use ($filters) {
            foreach ($filters as $field => $value) {
                if ($value === null || $value === '') {
                    continue;
                }

                switch ($field) {
                    case 'titulo':
                    case 'numero_certificado':
                        if (stripos($obra->$field, $value) === false) {
                            return false;
                        }
                        break;
                    case 'obra_intelectual':
                    case 'provincia_id':
                    case 'genero_id':
                    case 'classificacao_id':
                        if ($obra->$field != $value) {
                            return false;
                        }
                        break;
                }
            }
            return true;
        });
    }

    private function applySorting(array $obras, string $sort, string $order): array
    {
        usort($obras, function ($a, $b) use ($sort, $order) {
            $valueA = $a->$sort ?? '';
            $valueB = $b->$sort ?? '';

            if ($valueA == $valueB) {
                return 0;
            }

            $result = $valueA < $valueB ? -1 : 1;
            return $order === 'desc' ? -$result : $result;
        });

        return $obras;
    }

    private function processDocumentos(Request $request): array
    {
        $documentos = [];

        $allFiles = $request->allFiles();

        foreach ($allFiles as $key => $file) {
            if (preg_match('/^documentos\[(\d+)\]\[file\]$/', $key, $matches)) {
                $index = $matches[1];
                $tipoDocumentoId = $request->input("documentos[{$index}][tipo_documento_id]", 1);

                if ($file && $file->isValid()) {
                    $documentos[] = [
                        'tipo_documento_id' => (int) $tipoDocumentoId,
                        'file' => $file
                    ];
                }
            }
        }

        if (empty($documentos)) {
            $index = 0;
            while ($request->hasFile("documento_{$index}")) {
                $file = $request->file("documento_{$index}");
                $tipoDocumentoId = $request->input("tipo_documento_{$index}", 1);

                if ($file && $file->isValid()) {
                    $documentos[] = [
                        'tipo_documento_id' => (int) $tipoDocumentoId,
                        'file' => $file
                    ];
                }
                $index++;
            }
        }

        if (empty($documentos) && $request->hasFile('documento')) {
            $file = $request->file('documento');
            $tipoDocumentoId = $request->input('tipo_documento_id', 1);

            if ($file && $file->isValid()) {
                $documentos[] = [
                    'tipo_documento_id' => (int) $tipoDocumentoId,
                    'file' => $file
                ];
            }
        }

        return $documentos;
    }

    /**
     * @OA\Get(
     *     path="/api/v1/obras/{id}",
     *     summary="Obter obra por ID",
     *     description="Retorna os detalhes completos de uma obra específica incluindo autores e vínculos",
     *     tags={"Obras"},
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="ID único da obra",
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Obra encontrada com sucesso",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(
     *                 property="data",
     *                 type="object",
     *                 @OA\Property(property="id", type="integer", example=1),
     *                 @OA\Property(property="titulo", type="string", example="O Meu Livro"),
     *                 @OA\Property(property="numero_certificado", type="string", example="CERT-2024-001"),
     *                 @OA\Property(property="data_registo", type="string", format="date", example="2024-01-15"),
     *                 @OA\Property(property="provincia_id", type="integer", example=1),
     *                 @OA\Property(property="genero_id", type="integer", example=1),
     *                 @OA\Property(property="suporte", type="string", example="Papel"),
     *                 @OA\Property(property="classificacao_id", type="integer", example=1),
     *                 @OA\Property(property="esta_publicada", type="boolean", example=false),
     *                 @OA\Property(property="descricao", type="string", example="Uma obra literária"),
     *                 @OA\Property(property="observacoes", type="string", example="Primeira edição"),
     *                 @OA\Property(property="created_at", type="string", format="date-time"),
     *                 @OA\Property(property="updated_at", type="string", format="date-time")
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=404,
     *         description="Obra não encontrada",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="message", type="string", example="Obra não encontrada")
     *         )
     *     ),
     *     @OA\Response(
     *         response=500,
     *         description="Erro interno do servidor",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="message", type="string", example="Erro ao buscar obra")
     *         )
     *     )
     * )
     */
    public function show(int $id): JsonResponse
    {
        try {
            $obra = $this->obraRepository->findById($id);

            if (!$obra) {
                return response()->json([
                    'success' => false,
                    'message' => 'Obra não encontrada'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $obra
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao buscar obra: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * @OA\Get(
     *     path="/api/v1/obras/autor/{autorId}",
     *     summary="Listar obras por autor",
     *     description="Retorna todas as obras associadas a um autor específico, incluindo informações sobre o vínculo do autor em cada obra",
     *     tags={"Obras"},
     *     @OA\Parameter(
     *         name="autorId",
     *         in="path",
     *         required=true,
     *         description="ID único do autor",
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *     @OA\Parameter(
     *         name="page",
     *         in="query",
     *         required=false,
     *         description="Número da página para paginação",
     *         @OA\Schema(type="integer", default=1)
     *     ),
     *     @OA\Parameter(
     *         name="per_page",
     *         in="query",
     *         required=false,
     *         description="Número de itens por página",
     *         @OA\Schema(type="integer", default=15)
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Obras do autor retornadas com sucesso",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(
     *                 property="data",
     *                 type="array",
     *                 @OA\Items(
     *                     @OA\Property(property="id", type="integer", example=1),
     *                     @OA\Property(property="titulo", type="string", example="O Meu Livro"),
     *                     @OA\Property(property="numero_certificado", type="string", example="CERT-2024-001"),
     *                     @OA\Property(property="data_registo", type="string", format="date", example="2024-01-15"),
     *                     @OA\Property(property="provincia", type="string", example="Luanda"),
     *                     @OA\Property(property="genero_id", type="integer", example=1),
     *                     @OA\Property(property="suporte", type="string", example="Papel"),
     *                     @OA\Property(property="classificacao_id", type="integer", example=1),
     *                     @OA\Property(property="esta_publicada", type="boolean", example=false),
     *                     @OA\Property(property="descricao", type="string", example="Uma obra literária"),
     *                     @OA\Property(property="observacoes", type="string", example="Primeira edição"),
     *                     @OA\Property(property="vinculo", type="string", example="Autor Principal"),
     *                     @OA\Property(property="vinculo_observacoes", type="string", example="Responsável pela pesquisa"),
     *                     @OA\Property(property="created_at", type="string", format="date-time"),
     *                     @OA\Property(property="updated_at", type="string", format="date-time")
     *                 )
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=404,
     *         description="Autor não encontrado",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="message", type="string", example="Autor não encontrado")
     *         )
     *     ),
     *     @OA\Response(
     *         response=500,
     *         description="Erro interno do servidor",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="message", type="string", example="Erro ao buscar obras do autor")
     *         )
     *     )
     * )
     */
    public function findByAuthor(int $autorId): JsonResponse
    {
        try {
            $obras = $this->obraRepository->findByAuthor($autorId);

            return response()->json([
                'success' => true,
                'data' => $obras
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao buscar obras do autor: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * @OA\Post(
     *     path="/api/v1/obras/{id}/documentos",
     *     summary="Adicionar documento a uma obra",
     *     description="Adiciona um novo documento a uma obra existente",
     *     tags={"Obras"},
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="ID da obra",
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\MediaType(
     *             mediaType="multipart/form-data",
     *             @OA\Schema(
     *                 required={"file", "tipo_documento_id"},
     *                 @OA\Property(
     *                     property="file",
     *                     type="string",
     *                     format="binary",
     *                     description="Ficheiro do documento (máximo 10MB)"
     *                 ),
     *                 @OA\Property(
     *                     property="tipo_documento_id",
     *                     type="integer",
     *                     description="ID do tipo de documento (1-8)",
     *                     example=1
     *                 ),
     *                 @OA\Property(
     *                     property="ordem",
     *                     type="integer",
     *                     description="Ordem do documento (opcional)",
     *                     example=1
     *                 )
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=201,
     *         description="Documento adicionado com sucesso",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="message", type="string", example="Documento adicionado com sucesso"),
     *             @OA\Property(
     *                 property="data",
     *                 type="object",
     *                 @OA\Property(property="documento_id", type="integer", example=1),
     *                 @OA\Property(property="obra_id", type="integer", example=1),
     *                 @OA\Property(property="nome", type="string", example="documento.pdf"),
     *                 @OA\Property(property="tipo_documento_id", type="integer", example=1),
     *                 @OA\Property(property="ordem", type="integer", example=1),
     *                 @OA\Property(property="caminho", type="string", example="obras/1/documento.pdf")
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=404,
     *         description="Obra não encontrada",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="message", type="string", example="Obra não encontrada")
     *         )
     *     ),
     *     @OA\Response(
     *         response=422,
     *         description="Dados de validação inválidos",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="message", type="string", example="Dados de validação inválidos"),
     *             @OA\Property(property="errors", type="object")
     *         )
     *     )
     * )
     */
    public function addDocumento(Request $request, int $id): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'file' => 'required|file|max:10240',
                'tipo_documento_id' => 'required|integer|exists:tipo_documentos,id',
                'ordem' => 'nullable|integer|min:1'
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $dto = new AddDocumentoObraDTO(
                $id,
                $request->file('file'),
                $request->input('tipo_documento_id'),
                $request->input('ordem')
            );

            $result = $this->addDocumentoObraUseCase->execute($dto);

            return response()->json([
                'success' => true,
                'message' => 'Documento adicionado com sucesso',
                'data' => $result
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Dados de validação inválidos',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao adicionar documento: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * @OA\Post(
     *     path="/api/v1/obras/{id}/autores",
     *     summary="Adicionar autor a uma obra",
     *     description="Adiciona um novo autor a uma obra existente",
     *     tags={"Obras"},
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="ID da obra",
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\MediaType(
     *             mediaType="application/json",
     *             @OA\Schema(
     *                 required={"autor_id", "vinculo_id"},
     *                 @OA\Property(
     *                     property="autor_id",
     *                     type="integer",
     *                     description="ID do autor",
     *                     example=1
     *                 ),
     *                 @OA\Property(
     *                     property="vinculo_id",
     *                     type="integer",
     *                     description="ID do tipo de vínculo",
     *                     example=1
     *                 ),
     *                 @OA\Property(
     *                     property="observacoes",
     *                     type="string",
     *                     description="Observações sobre o vínculo (opcional)",
     *                     example="Autor principal da obra"
     *                 )
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=201,
     *         description="Autor adicionado com sucesso",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="message", type="string", example="Autor adicionado com sucesso"),
     *             @OA\Property(
     *                 property="data",
     *                 type="object",
     *                 @OA\Property(property="id", type="integer", example=1),
     *                 @OA\Property(property="autor_id", type="integer", example=1),
     *                 @OA\Property(property="obra_id", type="integer", example=1),
     *                 @OA\Property(property="vinculo_id", type="integer", example=1),
     *                 @OA\Property(property="observacoes", type="string", example="Autor principal da obra"),
     *                 @OA\Property(property="autor_nome", type="string", example="João Silva"),
     *                 @OA\Property(property="vinculo_nome", type="string", example="Autor Principal")
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=404,
     *         description="Obra, autor ou vínculo não encontrado",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="message", type="string", example="Obra não encontrada")
     *         )
     *     ),
     *     @OA\Response(
     *         response=422,
     *         description="Dados de validação inválidos ou autor já associado",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="message", type="string", example="O autor já está associado a esta obra"),
     *             @OA\Property(property="errors", type="object")
     *         )
     *     )
     * )
     */
    public function addAutor(Request $request, int $id): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'autor_id' => 'required|integer|exists:autores,id',
                'vinculo_id' => 'required|integer|exists:vinculos,id',
                'observacoes' => 'nullable|string|max:500'
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $dto = new AddAutorObraDTO(
                $id,
                $request->input('autor_id'),
                $request->input('vinculo_id'),
                $request->input('observacoes')
            );

            $result = $this->addAutorObraUseCase->execute($dto);

            return response()->json([
                'success' => true,
                'message' => 'Autor adicionado com sucesso',
                'data' => $result
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Dados de validação inválidos',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao adicionar autor: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * @OA\Post(
     *     path="/api/v1/obras/{id}/publicacao",
     *     summary="Adicionar/Atualizar dados de publicação de uma obra",
     *     description="Adiciona ou atualiza os dados de publicação de uma obra que está como PUBLICADA",
     *     tags={"Obras"},
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="ID da obra",
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\MediaType(
     *             mediaType="application/json",
     *             @OA\Schema(
     *                 @OA\Property(
     *                     property="editora",
     *                     type="string",
     *                     description="Nome da editora",
     *                     example="Editora Nacional"
     *                 ),
     *                 @OA\Property(
     *                     property="num_edicao",
     *                     type="string",
     *                     description="Número da edição",
     *                     example="1ª Edição"
     *                 ),
     *                 @OA\Property(
     *                     property="ano",
     *                     type="integer",
     *                     description="Ano de publicação",
     *                     example=2024
     *                 ),
     *                 @OA\Property(
     *                     property="produtora",
     *                     type="string",
     *                     description="Nome da produtora",
     *                     example="Produtora ABC"
     *                 ),
     *                 @OA\Property(
     *                     property="local_publicacao",
     *                     type="string",
     *                     description="Local de publicação",
     *                     example="Luanda, Angola"
     *                 ),
     *                 @OA\Property(
     *                     property="volume",
     *                     type="string",
     *                     description="Volume da obra",
     *                     example="Volume 1"
     *                 )
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=201,
     *         description="Dados de publicação adicionados/atualizados com sucesso",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="message", type="string", example="Dados de publicação atualizados com sucesso"),
     *             @OA\Property(
     *                 property="data",
     *                 type="object",
     *                 @OA\Property(property="id", type="integer", example=1),
     *                 @OA\Property(property="obra_id", type="integer", example=1),
     *                 @OA\Property(property="editora", type="string", example="Editora Nacional"),
     *                 @OA\Property(property="num_edicao", type="string", example="1ª Edição"),
     *                 @OA\Property(property="ano", type="integer", example=2024),
     *                 @OA\Property(property="produtora", type="string", example="Produtora ABC"),
     *                 @OA\Property(property="local_publicacao", type="string", example="Luanda, Angola"),
     *                 @OA\Property(property="volume", type="string", example="Volume 1"),
     *                 @OA\Property(property="created_at", type="string", format="date-time"),
     *                 @OA\Property(property="updated_at", type="string", format="date-time")
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=404,
     *         description="Obra não encontrada",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="message", type="string", example="Obra não encontrada")
     *         )
     *     ),
     *     @OA\Response(
     *         response=422,
     *         description="Obra não está como PUBLICADA ou dados inválidos",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="message", type="string", example="Apenas obras PUBLICADAS podem ter dados de publicação")
     *         )
     *     )
     * )
     */
    public function managePublicacao(Request $request, int $id): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'editora' => 'nullable|string|max:255',
                'num_edicao' => 'nullable|string|max:50',
                'ano' => 'nullable|integer|min:1900|max:' . date('Y'),
                'produtora' => 'nullable|string|max:255',
                'local_publicacao' => 'nullable|string|max:255',
                'volume' => 'nullable|string|max:50'
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $dto = new ObraPublicacaoDTO(
                $id,
                $request->input('editora'),
                $request->input('num_edicao'),
                $request->input('ano'),
                $request->input('produtora'),
                $request->input('local_publicacao'),
                $request->input('volume')
            );

            $result = $this->manageObraPublicacaoUseCase->execute($dto);

            return response()->json([
                'success' => true,
                'message' => 'Dados de publicação atualizados com sucesso',
                'data' => $result
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Dados de validação inválidos',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao atualizar dados de publicação: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * @OA\Get(
     *     path="/api/v1/obras/{id}/publicacao",
     *     summary="Obter dados de publicação de uma obra",
     *     description="Retorna os dados de publicação de uma obra específica",
     *     tags={"Obras"},
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="ID da obra",
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Dados de publicação retornados com sucesso",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(
     *                 property="data",
     *                 type="object",
     *                 @OA\Property(property="id", type="integer", example=1),
     *                 @OA\Property(property="obra_id", type="integer", example=1),
     *                 @OA\Property(property="editora", type="string", example="Editora Nacional"),
     *                 @OA\Property(property="num_edicao", type="string", example="1ª Edição"),
     *                 @OA\Property(property="ano", type="integer", example=2024),
     *                 @OA\Property(property="produtora", type="string", example="Produtora ABC"),
     *                 @OA\Property(property="local_publicacao", type="string", example="Luanda, Angola"),
     *                 @OA\Property(property="volume", type="string", example="Volume 1"),
     *                 @OA\Property(property="created_at", type="string", format="date-time"),
     *                 @OA\Property(property="updated_at", type="string", format="date-time")
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=404,
     *         description="Obra não encontrada ou sem dados de publicação",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="message", type="string", example="Dados de publicação não encontrados")
     *         )
     *     )
     * )
     */
    public function getPublicacao(int $id): JsonResponse
    {
        try {
            $publicacao = $this->manageObraPublicacaoUseCase->getByObraId($id);

            if (!$publicacao) {
                return response()->json([
                    'success' => false,
                    'message' => 'Dados de publicação não encontrados'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $publicacao
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao buscar dados de publicação: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * @OA\Delete(
     *     path="/api/v1/obras/{id}/documentos/{documento_id}",
     *     summary="Remover documento de uma obra",
     *     description="Remove completamente um documento de uma obra, incluindo o ficheiro físico e o registo da base de dados",
     *     tags={"Obras"},
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="ID da obra",
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *     @OA\Parameter(
     *         name="documento_id",
     *         in="path",
     *         required=true,
     *         description="ID do documento",
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Documento removido completamente com sucesso",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="message", type="string", example="Documento removido completamente com sucesso"),
     *             @OA\Property(
     *                 property="data",
     *                 type="object",
     *                 @OA\Property(property="success", type="boolean", example=true),
     *                 @OA\Property(property="message", type="string", example="Documento removido completamente com sucesso"),
     *                 @OA\Property(property="obra_id", type="integer", example=1),
     *                 @OA\Property(property="documento_id", type="integer", example=1),
     *                 @OA\Property(property="documento_nome", type="string", example="documento.pdf")
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=404,
     *         description="Obra ou documento não encontrado",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="message", type="string", example="Obra não encontrada")
     *         )
     *     ),
     *     @OA\Response(
     *         response=422,
     *         description="Documento não está associado à obra",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="message", type="string", example="Documento não está associado a esta obra")
     *         )
     *     )
     * )
     */
    public function removeDocumento(int $id, int $documentoId): JsonResponse
    {
        try {
            $result = $this->removeDocumentoObraUseCase->execute($id, $documentoId);

            return response()->json([
                'success' => true,
                'message' => 'Documento removido completamente com sucesso',
                'data' => $result
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao remover documento: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * @OA\Delete(
     *     path="/api/v1/obras/{id}/autores/{autor_id}",
     *     summary="Remover autor de uma obra",
     *     description="Remove um autor específico de uma obra. Não é possível remover o último autor.",
     *     tags={"Obras"},
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="ID da obra",
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *     @OA\Parameter(
     *         name="autor_id",
     *         in="path",
     *         required=true,
     *         description="ID do autor",
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Autor removido com sucesso",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="message", type="string", example="Autor removido com sucesso"),
     *             @OA\Property(
     *                 property="data",
     *                 type="object",
     *                 @OA\Property(property="success", type="boolean", example=true),
     *                 @OA\Property(property="message", type="string", example="Autor removido com sucesso"),
     *                 @OA\Property(property="obra_id", type="integer", example=1),
     *                 @OA\Property(property="autor_id", type="integer", example=1),
     *                 @OA\Property(property="autor_nome", type="string", example="João Silva")
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=404,
     *         description="Obra ou autor não encontrado",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="message", type="string", example="Obra não encontrada")
     *         )
     *     ),
     *     @OA\Response(
     *         response=422,
     *         description="Autor não está associado à obra ou é o último autor",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="message", type="string", example="Não é possível remover o último autor da obra")
     *         )
     *     )
     * )
     */
    public function removeAutor(int $id, int $autorId): JsonResponse
    {
        try {
            $result = $this->removeAutorObraUseCase->execute($id, $autorId);

            return response()->json([
                'success' => true,
                'message' => 'Autor removido com sucesso',
                'data' => $result
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao remover autor: ' . $e->getMessage()
            ], 500);
        }
    }
}
