<?php

namespace App\UI\Http\Controllers;

use App\Http\Controllers\Controller;
use Application\UseCases\DeleteAuthorDocument as UseCasesDeleteAuthorDocument;
use Illuminate\Http\JsonResponse;

class DocumentController extends Controller
{
    public function __construct(
        private UseCasesDeleteAuthorDocument $deleteAuthorDocument,
    ) {}
/**
 * @OA\Delete(
 *     path="/api/v1/documentos/{documentoId}/autor/{autorId}",
 *     summary="Elimina um documento associado a um autor",
 *     description="Remove o documento da base de dados e também o ficheiro do servidor, caso pertença ao autor.",
 *     operationId="deleteDocumentoAutor",
 *     tags={"Documentos"},
 *     @OA\Parameter(
 *         name="documentoId",
 *         in="path",
 *         required=true,
 *         description="ID do documento a ser eliminado",
 *         @OA\Schema(type="integer", example=1)
 *     ),
 *     @OA\Parameter(
 *         name="autorId",
 *         in="path",
 *         required=true,
 *         description="ID do autor a quem o documento pertence",
 *         @OA\Schema(type="integer", example=3)
 *     ),
 *     @OA\Response(
 *         response=200,
 *         description="Documento eliminado com sucesso",
 *         @OA\JsonContent(
 *             type="object",
 *             @OA\Property(property="message", type="string", example="Documento eliminado com sucesso")
 *         )
 *     ),
 *     @OA\Response(
 *         response=404,
 *         description="Documento ou associação não encontrada",
 *         @OA\JsonContent(
 *             type="object",
 *             @OA\Property(property="error", type="string", example="Documento Não Encontrado")
 *         )
 *     ),
 *     @OA\Response(
 *         response=500,
 *         description="Erro interno ao eliminar o documento"
 *     )
 * )
 */
    public function deleteAuthor(int $authorId, int $documentId): JsonResponse
    {

        try {
            $this->deleteAuthorDocument->execute($documentId, $authorId);

            return response()->json([
                'message' => 'Documento eliminado com sucesso.',
            ], 200);
        } catch (\DomainException $e) {
            return response()->json(['error' => $e->getMessage()], 400);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }
}
