<?php

namespace UI\Http\Controllers;

use App\Http\Controllers\Controller;
use Application\UseCases\{
    CreateClassificacao,
    DeleteClassificacao,
    FindClassificacao,
    ListClassificacoes,
    UpdateClassificacao
};
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use UI\Http\Requests\StoreClassificacaoRequest;
use UI\Http\Requests\UpdateClassificacaoRequest;

class ClassificacaoController extends Controller
{
    public function __construct(
        private CreateClassificacao $createUC,
        private UpdateClassificacao $updateUC,
        private DeleteClassificacao $deleteUC,
        private FindClassificacao $findUC,
        private ListClassificacoes $listUC
    ) {
    }

    /**
     * @OA\Get(
     *     path="/api/v1/classificacoes",
     *     tags={"Classificações"},
     *     summary="Listar classificações com paginação e filtro opcional",
     *     description="Retorna uma lista de classificações. Pode ser filtrada por designação e paginada.",
     *
     *     @OA\Parameter(
     *         name="search",
     *         in="query",
     *         description="Texto opcional para filtrar pela designação da classificação",
     *         required=false,
     *
     *         @OA\Schema(type="string", example="Romance")
     *     ),
     *
     *     @OA\Parameter(
     *         name="paginate",
     *         in="query",
     *         description="Define se deve paginar (true/false)",
     *         required=false,
     *
     *         @OA\Schema(type="boolean", example=true)
     *     ),
     *
     *     @OA\Parameter(
     *         name="page",
     *         in="query",
     *         description="Número da página atual",
     *         required=false,
     *
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *
     *     @OA\Parameter(
     *         name="perPage",
     *         in="query",
     *         description="Quantidade de resultados por página",
     *         required=false,
     *
     *         @OA\Schema(type="integer", example=10)
     *     ),
     *
     *     @OA\Response(
     *         response=200,
     *         description="Lista de classificações retornada com sucesso",
     *
     *         @OA\JsonContent(
     *             type="object",
     *
     *             @OA\Property(property="data", type="array",
     *
     *                 @OA\Items(
     *
     *                     @OA\Property(property="id", type="integer", example=1),
     *                     @OA\Property(property="designacao", type="string", example="Romance"),
     *                     @OA\Property(property="genero_id", type="integer", example=1),
     *                     @OA\Property(property="created_at", type="string", format="date-time"),
     *                     @OA\Property(property="updated_at", type="string", format="date-time")
     *                 )
     *             ),
     *             @OA\Property(property="meta", type="object",
     *                 @OA\Property(property="current_page", type="integer", example=1),
     *                 @OA\Property(property="per_page", type="integer", example=10),
     *                 @OA\Property(property="total", type="integer", example=30),
     *                 @OA\Property(property="last_page", type="integer", example=3)
     *             )
     *         )
     *     )
     * )
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $page = (int) $request->input('page', 1);
            $perPage = (int) $request->input('perPage', 20);
            $search = $request->input('search') ?? $request->input('nome') ?? $request->input('descricao');

            $result = $this->listUC->execute($perPage, $page, $search, true);

            return response()->json([
                'success' => true,
                'data' => $result,
                'pagination' => [
                    'current_page' => $page,
                    'per_page' => $perPage,
                    'total' => count($result),
                    'last_page' => ceil(count($result) / $perPage)
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * @OA\Post(
     *     path="/api/v1/classificacoes",
     *     tags={"Classificações"},
     *     summary="Criar nova classificação",
     *     description="Regista uma nova classificação associada a um género.",
     *
     *     @OA\RequestBody(
     *         required=true,
     *
     *         @OA\JsonContent(
     *             required={"designacao","genero_id"},
     *
     *             @OA\Property(property="designacao", type="string", example="Romance"),
     *             @OA\Property(property="genero_id", type="integer", example=1)
     *         )
     *     ),
     *
     *     @OA\Response(
     *         response=201,
     *         description="Classificação criada com sucesso",
     *
     *         @OA\JsonContent(
     *
     *             @OA\Property(property="id", type="integer", example=1),
     *             @OA\Property(property="designacao", type="string", example="Romance"),
     *             @OA\Property(property="genero_id", type="integer", example=1)
     *         )
     *     )
     * )
     */
    public function store(StoreClassificacaoRequest $request): JsonResponse
    {
        $classificacao = $this->createUC->execute($request->designacao, $request->genero_id);
        return response()->json($classificacao, 201);
    }

    /**
     * @OA\Get(
     *     path="/api/v1/classificacoes/{id}",
     *     tags={"Classificações"},
     *     summary="Obter detalhes de uma classificação",
     *     description="Retorna os dados de uma classificação pelo seu ID.",
     *
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *
     *     @OA\Response(
     *         response=200,
     *         description="Classificação encontrada",
     *
     *         @OA\JsonContent(
     *
     *             @OA\Property(property="id", type="integer", example=1),
     *             @OA\Property(property="designacao", type="string", example="Romance"),
     *             @OA\Property(property="genero_id", type="integer", example=1)
     *         )
     *     ),
     *
     *     @OA\Response(response=404, description="Classificação não encontrada")
     * )
     */
    public function show(string $id): JsonResponse
    {
        $classificacao = $this->findUC->execute($id);
        return $classificacao
            ? response()->json($classificacao)
            : response()->json(['message' => 'Classificação não encontrada'], 404);
    }

    /**
     * @OA\Put(
     *     path="/api/v1/classificacoes/{id}",
     *     tags={"Classificações"},
     *     summary="Actualizar uma classificação existente",
     *     description="Actualiza os dados de uma classificação pelo ID.",
     *
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *
     *     @OA\RequestBody(
     *         required=true,
     *
     *         @OA\JsonContent(
     *             required={"designacao","genero_id"},
     *
     *             @OA\Property(property="designacao", type="string", example="Drama"),
     *             @OA\Property(property="genero_id", type="integer", example=1)
     *         )
     *     ),
     *
     *     @OA\Response(
     *         response=200,
     *         description="Classificação actualizada com sucesso",
     *
     *         @OA\JsonContent(
     *
     *             @OA\Property(property="id", type="integer", example=1),
     *             @OA\Property(property="designacao", type="string", example="Drama"),
     *             @OA\Property(property="genero_id", type="integer", example=1)
     *         )
     *     )
     * )
     */
    public function update(UpdateClassificacaoRequest $request, string $id): JsonResponse
    {
        $classificacao = $this->updateUC->execute($id, $request->designacao, $request->genero_id);
        return response()->json($classificacao);
    }

    /**
     * @OA\Delete(
     *     path="/api/v1/classificacoes/{id}",
     *     tags={"Classificações"},
     *     summary="Remover uma classificação",
     *     description="Apaga uma classificação existente pelo seu ID.",
     *
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *
     *     @OA\Response(response=200, description="Classificação removida com sucesso"),
     *     @OA\Response(response=404, description="Classificação não encontrada")
     * )
     */
    public function destroy(string $id): JsonResponse
    {
        $deleted = $this->deleteUC->execute($id);
        return response()->json(['deleted' => $deleted]);
    }
}
