<?php

namespace UI\Http\Controllers;

use App\Http\Controllers\Controller;
use Application\DTOs\AutorData;
use Application\DTOs\DocumentData;
use Application\Shared\Search\Paginate;
use Application\Shared\Search\Search;
use Application\Shared\Search\SearchWithPagination;
use Application\UseCases\CreateAuthor;
use Application\UseCases\CreateAutorDocument;
use Application\UseCases\CreateDocument;
use Application\UseCases\DeleteAuthor;
use Application\UseCases\UpdateAuthor;
use Domain\Entities\AutorRepresentanteLegal;
use Domain\Repositories\IAutorRepository;
use DomainException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use UI\Http\Requests\StoreAutorRequest;
use UI\Http\Requests\UpdateAutorRequest;

class AutorController extends Controller
{
    /**
     * @OA\Post(
     *     path="/api/v1/autores",
     *     summary="Cria um novo autor com documentos e representante legal",
     *     tags={"Autores"},
     *     operationId="storeAutor",
     *
     *     @OA\RequestBody(
     *         required=true,
     *
     *         @OA\MediaType(
     *             mediaType="multipart/form-data",
     *
     *             @OA\Schema(
     *                 required={"nome","email","numero_identificacao","provincia_id","telefone","data_nascimento"},
     *
     *                 @OA\Property(property="nome", type="string", example="João Silva"),
     *                 @OA\Property(property="email", type="string", example="joao@example.com"),
     *                 @OA\Property(property="provincia_id", type="integer", example=2),
     *                 @OA\Property(property="numero_identificacao", type="string", example="123456"),
     *                 @OA\Property(property="telefone", type="string", example="923456789"),
     *                 @OA\Property(property="data_nascimento", type="string", format="date", example="2000-10-01"),
     *                 @OA\Property(property="nacionalidade_id", type="integer", example=10),
     *                 @OA\Property(property="pseudonimo", type="string", example="J. Silva"),
     *                 @OA\Property(property="naturalidade", type="string", example="Luanda"),
     *                 @OA\Property(property="nome_pai", type="string", example="António Silva"),
     *                 @OA\Property(property="nome_mae", type="string", example="Maria Silva"),
     *                 @OA\Property(property="ocupacao", type="string", example="Professor"),
     *                 @OA\Property(property="endereco", type="string", example="Rua das Flores, nº 12"),
     *                 @OA\Property(property="bairro", type="string", example="Talatona"),
     *                 @OA\Property(property="municipio_id", type="integer", example="2"),
     *                 @OA\Property(property="celular", type="string", example="923888777"),
     *                 @OA\Property(property="site", type="string", example="https://joaosilva.ao"),
     *                 @OA\Property(
     *                     property="representante_nome",
     *                     type="string",
     *                     example="Maria José"
     *                 ),
     *                 @OA\Property(
     *                     property="representante_grau_parentesco",
     *                     type="string",
     *                     example="Mãe"
     *                 ),
     *                 @OA\Property(
     *                     property="representante_numero_identificacao",
     *                     type="string",
     *                     example="654321"
     *                 ),
     *                 @OA\Property(
     *                     property="documentos[0][file]",
     *                     type="string",
     *                     format="binary",
     *                     description="Ficheiro do documento"
     *                 ),
     *                 @OA\Property(
     *                     property="documentos[0][tipo_documento_id]",
     *                     type="integer",
     *                     example=1
     *                 ),
     *                 @OA\Property(
     *                     property="documentos[1][file]",
     *                     type="string",
     *                     format="binary",
     *                     description="Outro ficheiro de documento"
     *                 ),
     *                 @OA\Property(
     *                     property="documentos[1][tipo_documento_id]",
     *                     type="integer",
     *                     example=2
     *                 )
     *             )
     *         )
     *     ),
     *
     *     @OA\Response(
     *         response=201,
     *         description="Autor criado com sucesso",
     *
     *         @OA\JsonContent(
     *
     *             @OA\Property(property="message", type="string", example="Autor criado com sucesso"),
     *             @OA\Property(property="autor_id", type="integer", example=10)
     *         )
     *     ),
     *
     *     @OA\Response(
     *         response=422,
     *         description="Erro de validação",
     *
     *         @OA\JsonContent(
     *
     *             @OA\Property(property="message", type="string", example="Os dados fornecidos são inválidos."),
     *             @OA\Property(
     *                 property="errors",
     *                 type="object",
     *                 example={
     *                     "email": {"O campo email é obrigatório."},
     *                     "nome": {"O campo nome é obrigatório."}
     *                 }
     *             )
     *         )
     *     )
     * )
     */

    public function __construct(
        private IAutorRepository $repo,
        private CreateAuthor $createAutor,
        private CreateDocument $createDocument,
        private CreateAutorDocument $createAutorDocument,
        private DeleteAuthor $deleteAuthor,
        private UpdateAuthor $updateAuthor
    ) {
    }

    public function store(StoreAutorRequest $request): JsonResponse
    {
        try {

            $representanteData =  ["nome" => $request['representante_nome'],
            "grau_parentesco" => $request['representante_grau_parentesco'],
            "numero_identificacao" => $request['representante_numero_identificacao']
        ];

            $representanteEntity = null;

            if (! empty($representanteData) && isset($representanteData['nome'])) {
                $representanteEntity = new AutorRepresentanteLegal(
                    id: null,
                    nome: $representanteData['nome'],
                    grauParentesco: $representanteData['grau_parentesco'],
                    numeroIdentificacao: $representanteData['numero_identificacao'],
                );
            }

            $dtoAutor = new AutorData(
                nome: $request['nome'],
                email: $request['email'],
                provincia_id: $request['provincia_id'],
                numero_identificacao: $request['numero_identificacao'],
                telefone: $request['telefone'],
                data_nascimento: $request['data_nascimento'],
                nacionalidade_id: $request['nacionalidade_id'],
                pseudonimo: $request['pseudonimo'],
                naturalidade: $request['naturalidade'],
                nome_pai: $request['nome_pai'],
                nome_mae: $request['nome_mae'],
                ocupacao: $request['ocupacao'],
                endereco: $request['endereco'],
                bairro: $request['bairro'],
                municipio_id: $request['municipio_id'],
                celular: $request['celular'],
                site: $request['site'],
                representante_legal: $representanteEntity
            );



            $author = $this->createAutor->execute($dtoAutor);

            if ($author) {
                foreach ($request->input('documentos', []) as $index => $docData) {
                    $docFile = $request->file("documentos.$index.file");
                    $tipoId = $docData['tipo_documento_id'] ?? null;

                    if ($docFile instanceof \Illuminate\Http\UploadedFile) {
                        $path = $docFile->store('documentos');

                        $documento = new DocumentData(
                            nome: $docFile->getClientOriginalName(),
                            caminho_arquivo: $path,
                            tamanho: $docFile->getSize(),
                            extensao: $docFile->getClientOriginalExtension(),
                            tipo_documento_id: $tipoId,
                        );

                        $documentCreated =    $this->createDocument->execute($documento);
                        $this->createAutorDocument->execute($author, $documentCreated);
                    }
                }
            }

            return response()->json([
                'message' => 'Autor criado com sucesso',
                'data' => $author,
            ], 201);
        } catch (DomainException $e) {
            return response()->json(['error' => $e->getMessage()], 400);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }
    /**
     * @OA\Get(
     *     path="/api/v1/autores",
     *     summary="Lista os autores com suporte a filtros, ordenação e paginação",
     *     tags={"Autores"},
     *     operationId="listAutores",
     *
     *     @OA\Parameter(
     *         name="page",
     *         in="query",
     *         description="Número da página atual",
     *         required=false,
     *
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *
     *     @OA\Parameter(
     *         name="perPage",
     *         in="query",
     *         description="Quantidade de resultados por página",
     *         required=false,
     *
     *         @OA\Schema(type="integer", example=20)
     *     ),
     *
     *     @OA\Parameter(
     *         name="sort",
     *         in="query",
     *         description="Campo usado para ordenação (ex: nome, email, created_at)",
     *         required=false,
     *
     *         @OA\Schema(type="string", example="updated_at")
     *     ),
     *
     *     @OA\Parameter(
     *         name="direction",
     *         in="query",
     *         description="Direção da ordenação (asc ou desc)",
     *         required=false,
     *
     *         @OA\Schema(type="string", enum={"asc", "desc"}, example="desc")
     *     ),
     *
     *     @OA\Parameter(
     *         name="nome",
     *         in="query",
     *         description="Filtra autores pelo nome (usa operação LIKE)",
     *         required=false,
     *
     *         @OA\Schema(type="string", example="João")
     *     ),
     *
     *     @OA\Parameter(
     *         name="telefone",
     *         in="query",
     *         description="Filtra autores pelo telefone (usa operação LIKE)",
     *         required=false,
     *
     *         @OA\Schema(type="string", example="923456789")
     *     ),
     *
     *     @OA\Parameter(
     *         name="email",
     *         in="query",
     *         description="Filtra autores pelo email (usa operação LIKE)",
     *         required=false,
     *
     *         @OA\Schema(type="string", example="joao@example.com")
     *     ),
     *
     *     @OA\Response(
     *         response=200,
     *         description="Lista de autores retornada com sucesso",
     *
     *         @OA\JsonContent(
     *
     *             @OA\Property(property="data", type="array",
     *
     *                 @OA\Items(
     *
     *                     @OA\Property(property="id", type="integer", example=10),
     *                     @OA\Property(property="nome", type="string", example="João Silva"),
     *                     @OA\Property(property="email", type="string", example="joao@example.com"),
     *                     @OA\Property(property="telefone", type="string", example="923456789"),
     *                     @OA\Property(property="provincia_id", type="integer", example=1),
     *                     @OA\Property(property="data_nascimento", type="string", example="2000-10-01"),
     *                     @OA\Property(property="created_at", type="string", example="2025-10-09T12:00:00Z"),
     *                     @OA\Property(property="updated_at", type="string", example="2025-10-09T13:00:00Z")
     *                 )
     *             ),
     *             @OA\Property(
     *                 property="pagination",
     *                 type="object",
     *                 @OA\Property(property="current_page", type="integer", example=1),
     *                 @OA\Property(property="per_page", type="integer", example=20),
     *                 @OA\Property(property="total", type="integer", example=125),
     *                 @OA\Property(property="last_page", type="integer", example=7)
     *             )
     *         )
     *     ),
     *
     *     @OA\Response(
     *         response=400,
     *         description="Parâmetros inválidos na requisição",
     *
     *         @OA\JsonContent(
     *
     *             @OA\Property(property="message", type="string", example="Parâmetros inválidos fornecidos.")
     *         )
     *     )
     * )
     */
    public function index(Request $request)
    {
        $page = (int) $request->input('page', 1);
        $perPage = (int) $request->input('perPage', 20);
        $sort = $request->input('sort', 'updated_at');
        $order = $request->input('direction', 'desc');

        $filters = [
            'nome' => [
                'operation' => 'like',
                'value' => $request->input('nome'),
            ],
            'telefone' => ['operation' => 'like', 'value' => $request->input('telefone')],
            'email' => ['operation' => 'like', 'value' => $request->input('email')],
        ];

        $search = new Search([
            'orderBy' => 'created_at',
            'orderByDirection' => 'asc',
            'filters' => $filters,
        ]);

        $paginate = new Paginate(['withPagination' => true, 'page' => $page, 'perPage' => $perPage]);

        $input = new SearchWithPagination($search, $paginate);
        $request = $this->repo->list($input);

        return response()->json($request->toArray());
    }
    /**
     * @OA\Get(
     *   path="/api/v1/autores/{id}",
     *   tags={"Autores"},
     *
     *   @OA\Parameter(name="id", in="path", required=true, @OA\Schema(type="integer")),
     *
     *   @OA\Response(response=200, description="Get author"),
     *   @OA\Response(response=404, description="Not found")
     * )
     */
    public function show(int $id): JsonResponse
    {
        $autor = $this->repo->findById($id);
        if (!$autor) {
            return response()->json(['message' => 'Autor não encontrado'], 404);
        }
        return response()->json(['data' => $autor->data]);
    }
    /**
     * @OA\Delete(
     *   path="/api/v1/autores/{id}",
     *   tags={"Autores"},
     *
     *   @OA\Parameter(name="id", in="path", required=true, @OA\Schema(type="integer")),
     *
     *   @OA\Response(response=204, description="Deleted")
     * )
     */
    public function destroy(int $id): JsonResponse
    {
        try {
            $this->deleteAuthor->execute($id);
            return response()->json(['message' => 'Autor excluído com sucesso'], 200);
        } catch (DomainException $e) {
            return response()->json(['message' => $e->getMessage()], 404);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * @OA\Put(
     *     path="/api/v1/autores/{id}",
     *     summary="Atualiza os dados de um autor e seu representante legal",
     *     description="Atualiza as informações de um autor existente e, opcionalmente, do seu representante legal associado.",
     *     tags={"Autores"},
     *     operationId="updateAutor",
     *
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="ID do autor a ser atualizado",
     *
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *
     *     @OA\RequestBody(
     *         required=true,
     *
     *         @OA\MediaType(
     *             mediaType="application/json",
     *
     *             @OA\Schema(
     *                 type="object",
     *                 required={"nome","email","numero_identificacao","provincia_id","telefone","data_nascimento"},
     *
     *                 @OA\Property(property="nome", type="string", example="João Silva"),
     *                 @OA\Property(property="email", type="string", example="joao@example.com"),
     *                 @OA\Property(property="numero_identificacao", type="string", example="123456"),
     *                 @OA\Property(property="provincia_id", type="integer", example=1),
     *                 @OA\Property(property="telefone", type="string", example="923456789"),
     *                 @OA\Property(property="data_nascimento", type="string", format="date", example="2000-10-01"),
     *                 @OA\Property(
     *                     property="representante",
     *                     type="object",
     *                     nullable=true,
     *                     description="Informações do representante legal (opcional)",
     *                     @OA\Property(property="nome", type="string", example="Maria José"),
     *                     @OA\Property(property="email", type="string", example="maria@example.com"),
     *                     @OA\Property(property="telefone", type="string", example="923111222"),
     *                     @OA\Property(property="grau_parentesco", type="string", example="Mãe"),
     *                     @OA\Property(property="provincia_id", type="integer", example=2),
     *                     @OA\Property(property="numero_identificacao", type="string", example="654321")
     *                 )
     *             )
     *         )
     *     ),
     *
     *     @OA\Response(
     *         response=200,
     *         description="Autor atualizado com sucesso",
     *
     *         @OA\JsonContent(
     *             type="object",
     *
     *             @OA\Property(property="message", type="string", example="Autor atualizado com sucesso"),
     *             @OA\Property(property="data", type="object",
     *                 @OA\Property(property="id", type="integer", example=10),
     *                 @OA\Property(property="nome", type="string", example="João Silva"),
     *                 @OA\Property(property="email", type="string", example="joao@example.com")
     *             )
     *         )
     *     ),
     *
     *     @OA\Response(
     *         response=404,
     *         description="Autor não encontrado",
     *
     *         @OA\JsonContent(
     *
     *             @OA\Property(property="message", type="string", example="Autor não encontrado")
     *         )
     *     ),
     *
     *     @OA\Response(
     *         response=422,
     *         description="Erro de validação dos dados enviados",
     *
     *         @OA\JsonContent(
     *             type="object",
     *
     *             @OA\Property(property="message", type="string", example="Os dados fornecidos são inválidos."),
     *             @OA\Property(
     *                 property="errors",
     *                 type="object",
     *                 example={
     *                     "email": {"O campo email é obrigatório."},
     *                     "nome": {"O campo nome é obrigatório."}
     *                 }
     *             )
     *         )
     *     )
     * )
     */
    public function update(UpdateAutorRequest $request, int $id): JsonResponse
    {
        try {
            $representanteData = $request->input('representante');

            $representanteEntity = null;

            if (!empty($representanteData) && isset($representanteData['nome'])) {
                $representanteEntity = new AutorRepresentanteLegal(
                    id: $representanteData['id'] ?? null,
                    nome: $representanteData['nome'],
                    grauParentesco: $representanteData['grau_parentesco'],
                    numeroIdentificacao: $representanteData['numero_identificacao']
                );
            }

            $dtoAutor = new AutorData(
                nome: $request->input('nome'),
                email: $request->input('email'),
                provincia_id: $request->input('provincia_id'),
                numero_identificacao: $request->input('numero_identificacao'),
                telefone: $request->input('telefone'),
                data_nascimento: $request->input('data_nascimento'),
                nacionalidade_id: $request->input('nacionalidade_id'),
                pseudonimo: $request->input('pseudonimo'),
                naturalidade: $request->input('naturalidade'),
                nome_pai: $request->input('nome_pai'),
                nome_mae: $request->input('nome_mae'),
                ocupacao: $request->input('ocupacao'),
                endereco: $request->input('endereco'),
                bairro: $request->input('bairro'),
                municipio_id: $request->input('municipio_id'),
                celular: $request->input('celular'),
                site: $request->input('site'),
                representante_legal: $representanteEntity
            );

            $autorAtualizado = $this->updateAuthor->execute($dtoAutor, $id);

            return response()->json([
                'message' => 'Autor atualizado com sucesso',
                'data' => $autorAtualizado,
            ], 200);
        } catch (DomainException $e) {
            return response()->json(['error' => $e->getMessage()], 400);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }







    }

    /**
     * @OA\Get(
     *   path="/api/v1/autores/all",
     *   tags={"Autores"},
     *
     *   @OA\Response(response=200, description="Get province"),
     *   @OA\Response(response=404, description="Not found")
     * )
     */
    public function drowpDown()
    {
        $search = new Search([]);

        $paginate = new Paginate(['withPagination' => false]);

        $input = new SearchWithPagination($search, $paginate);

        $result = $this->repo->list($input);

        return response()->json($result->toArray());
    }



}
