<?php

namespace Infrastructure\Persistence\Repositories;

use Domain\Entities\Classificacao;
use Domain\Entities\Genero;
use Domain\Repositories\IClassificacaoRepositoryInterface;
use Infrastructure\Persistence\Eloquent\Models\ClassificacaoModel;

class EloquentClassificacaoRepository implements IClassificacaoRepositoryInterface
{
    public function save(Classificacao $classificacao): Classificacao
    {
        $model = ClassificacaoModel::updateOrCreate(
            ['id' => $classificacao->id],
            [
                'designacao' => $classificacao->designacao,
                'genero_id' => $classificacao->genero_id,
            ]
        );

        return new Classificacao($model->designacao, $model->genero_id, $model->id);
    }

    public function findById(string $id): ?Classificacao
    {
        $model = ClassificacaoModel::find($id);
        if (!$model) {
            return null;
        }

        return new Classificacao(
            $model->designacao,
            $model->genero_id,
            $model->id,
            $model->relationLoaded('genero') ? new Genero(
                $model->genero->designacao,
                $model->genero->id
            ) : null
        );
    }

    public function delete(string $id): bool
    {
        return (bool) ClassificacaoModel::destroy($id);
    }

    public function all(int $perPage = 15, int $page = 1, ?string $search = null, bool $paginate = true): array
    {
        $query = ClassificacaoModel::with('genero');

        if ($search) {
            $query->where('designacao', 'ILIKE', "%{$search}%");
        }

        if ($paginate) {
            $paginator = $query->paginate($perPage, ['*'], 'page', $page);
            $items = [];
        } else {
            $items = $query->get()->toArray();
        }

        foreach ($paginator->items() as $m) {
            $items[] = new Classificacao(
                $m->designacao,
                $m->genero_id,
                $m->id,
                $m->relationLoaded('genero') ? new Genero(
                    $m->genero->designacao,
                    $m->genero->id
                ) : null
            );
        }

        return [
            'data' => $items,
            'meta' => [
                'current_page' => $paginator->currentPage(),
                'per_page' => $paginator->perPage(),
                'total' => $paginator->total(),
                'last_page' => $paginator->lastPage(),
            ]
        ];
    }
}
