<?php

namespace Domain\Entities;

use DomainException;

class Obra
{
    public ?int $id;
    public string $titulo;
    public string $numero_certificado;
    public \DateTime $data_registo;
    public int $provincia_id;
    public int $genero_id;
    public ?Genero $genero = null;
    public string $suporte;
    public int $classificacao_id;
    public ?Classificacao $classificacao = null;
    public string $obra_intelectual;
    public ?string $descricao = null;
    public ?string $observacoes = null;
    public \DateTimeImmutable $createdAt;
    public ?\DateTimeImmutable $updatedAt;

    public function __construct(
        ?int $id,
        string $titulo,
        string $numero_certificado,
        \DateTime $data_registo,
        int $provincia_id,
        int $genero_id,
        string $suporte,
        int $classificacao_id,
        string $obra_intelectual,
        ?string $descricao = null,
        ?string $observacoes = null,
        ?\DateTimeImmutable $createdAt = null,
        ?\DateTimeImmutable $updatedAt = null
    ) {
        // Validação da data
        $this->validateRegistrationDate($data_registo);

        $this->id = $id;
        $this->titulo = $titulo;
        $this->numero_certificado = $numero_certificado;
        $this->data_registo = $data_registo;
        $this->provincia_id = $provincia_id;
        $this->genero_id = $genero_id;
        $this->suporte = $suporte;
        $this->classificacao_id = $classificacao_id;
        $this->obra_intelectual = $obra_intelectual;
        $this->descricao = $descricao;
        $this->observacoes = $observacoes;
        $this->createdAt = $createdAt ?? new \DateTimeImmutable();
        $this->updatedAt = $updatedAt;
    }

    private function validateRegistrationDate(\DateTime $date): void
    {
        $hoje = new \DateTime();
        $hoje->setTime(0, 0, 0);

        $dateToCompare = clone $date;
        $dateToCompare->setTime(0, 0, 0);

        if ($dateToCompare > $hoje) {
            throw new DomainException(
                "A data de registro não pode ser maior que a data de hoje."
            );
        }
    }

    public function update(
        string $titulo,
        string $numero_certificado,
        \DateTime $data_registo,
        int $provincia_id,
        int $genero_id,
        string $suporte,
        int $classificacao_id,
        string $obra_intelectual,
        ?string $descricao = null,
        ?string $observacoes = null
    ): void {
        $this->validateRegistrationDate($data_registo);

        $this->titulo = $titulo;
        $this->numero_certificado = $numero_certificado;
        $this->data_registo = $data_registo;
        $this->provincia_id = $provincia_id;
        $this->genero_id = $genero_id;
        $this->suporte = $suporte;
        $this->classificacao_id = $classificacao_id;
        $this->obra_intelectual = $obra_intelectual;
        $this->descricao = $descricao;
        $this->observacoes = $observacoes;
        $this->updatedAt = new \DateTimeImmutable();
    }

    public function toArray(): array
    {
        return [
            'id' => $this->id,
            'titulo' => $this->titulo,
            'numero_certificado' => $this->numero_certificado,
            'data_registo' => $this->data_registo->format('Y-m-d'),
            'provincia_id' => $this->provincia_id,
            'genero_id' => $this->genero_id,
            'suporte' => $this->suporte,
            'classificacao_id' => $this->classificacao_id,
            'obra_intelectual' => $this->obra_intelectual,
            'descricao' => $this->descricao,
            'observacoes' => $this->observacoes,
            'created_at' => $this->createdAt->format('Y-m-d H:i:s'),
            'updated_at' => $this->updatedAt?->format('Y-m-d H:i:s')
        ];
    }
}
