<?php

namespace Application\UseCases;

use Application\DTOs\UpdateObraDTO;
use Domain\Entities\Obra;
use Domain\Entities\AutorObra;
use Domain\Entities\ObraPublicacao;
use Domain\Repositories\IObraRepository;
use Domain\Repositories\IAutorObraRepository;
use Domain\Repositories\IAutorRepository;
use Domain\Repositories\IVinculoRepositoryInterface;
use Domain\Repositories\IDocumentRepository;
use Domain\Repositories\IObraPublicacaoRepository;
use Infrastructure\Persistence\Eloquent\Models\ObraDocumentoModel;
use Domain\Services\IFileStorageService;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class UpdateObra
{
    public function __construct(
        private IObraRepository $obraRepository,
        private IAutorObraRepository $autorObraRepository,
        private IAutorRepository $autorRepository,
        private IVinculoRepositoryInterface $vinculoRepository,
        private IDocumentRepository $documentRepository,
        private IObraPublicacaoRepository $obraPublicacaoRepository,
        private IFileStorageService $fileStorageService
    ) {}

    public function execute(UpdateObraDTO $dto): Obra
    {
        return DB::transaction(function () use ($dto) {
            $existingObra = $this->obraRepository->findById($dto->id);
            if (!$existingObra) {
                throw new \InvalidArgumentException('Obra não encontrada');
            }

            $obraWithSameCertificate = $this->obraRepository->findByCertificateNumber($dto->numero_certificado);
            if ($obraWithSameCertificate && $obraWithSameCertificate->id !== $dto->id) {
                throw new \InvalidArgumentException('Já existe uma obra com este número de certificado.');
            }

            $this->validateAutores($dto->autores);

            $obra = new Obra(
                $dto->id,
                $dto->titulo,
                $dto->numero_certificado,
                new \DateTime($dto->data_registo),
                $dto->provincia_id,
                $dto->genero_id,
                $dto->suporte,
                $dto->classificacao_id,
                $dto->obra_intelectual,
                $dto->descricao,
                $dto->observacoes,
                $existingObra->createdAt,
                new \DateTimeImmutable()
            );

            $updatedObra = $this->obraRepository->update($obra);

            $this->updateAutores($dto->autores, $dto->id);

            if (!empty($dto->documentos)) {
                $this->processDocumentUploads($dto->documentos, $dto->id);
            }

            if ($dto->obra_intelectual === 'PUBLICADA') {
                $this->updateObraPublicacao($dto, $dto->id);
            } else {
                $this->obraPublicacaoRepository->deleteByObraId($dto->id);
            }

            $documentosExistentes = $this->getDocumentosExistentes($dto->id);

            return [
                'obra' => $updatedObra,
                'documentos_existentes' => $documentosExistentes
            ];
        });
    }

    private function validateAutores(array $autores): void
    {
        if (empty($autores)) {
            throw new \InvalidArgumentException('Uma obra deve ter pelo menos um autor.');
        }

        foreach ($autores as $autorData) {
            $autor = $this->autorRepository->findById($autorData['autor_id']);
            if (!$autor) {
                throw new \InvalidArgumentException("Autor com ID {$autorData['autor_id']} não encontrado.");
            }

            $vinculo = $this->vinculoRepository->findById($autorData['vinculo_id']);
            if (!$vinculo) {
                throw new \InvalidArgumentException("Vínculo com ID {$autorData['vinculo_id']} não encontrado.");
            }
        }
    }

    private function updateAutores(array $autores, int $obraId): void
    {
        $this->autorObraRepository->deleteByObra($obraId);

        foreach ($autores as $autorData) {
            $autorObra = new AutorObra(
                $autorData['autor_id'],
                $obraId,
                $autorData['vinculo_id'],
                $autorData['observacoes'] ?? null
            );

            $this->autorObraRepository->save($autorObra);
        }
    }

    private function processDocumentUploads(array $documentos, int $obraId): void
    {
        foreach ($documentos as $index => $documentoData) {
            if (isset($documentoData['file']) && $documentoData['file'] instanceof UploadedFile) {
                $file = $documentoData['file'];

                $fileName = Str::uuid() . '.' . $file->getClientOriginalExtension();

                $filePath = $this->fileStorageService->storeFile($file, 'obras/' . $obraId, $fileName);

                $document = new \Domain\Entities\Document(
                    null,
                    $file->getClientOriginalName(),
                    $filePath,
                    $documentoData['tipo_documento_id'] ?? 1,
                    $file->getSize(),
                    $file->getClientOriginalExtension()
                );

                $savedDocument = $this->documentRepository->save($document);

                $tipoDocumentoId = $documentoData['tipo_documento_id'] ?? 1;

                \DB::table('obra_documentos')->insert([
                    'obra_id' => $obraId,
                    'documento_id' => $savedDocument->id,
                    'tipo_documento_id' => $tipoDocumentoId,
                    'ordem' => $index + 1,
                    'created_at' => now(),
                    'updated_at' => now()
                ]);
            }
        }
    }

    private function updateObraPublicacao(UpdateObraDTO $dto, int $obraId): void
    {
        $existingPublicacao = $this->obraPublicacaoRepository->findByObraId($obraId);

        if ($existingPublicacao) {
            $existingPublicacao->update(
                $dto->editora,
                $dto->num_edicao,
                $dto->ano,
                $dto->produtora,
                $dto->local_publicacao,
                $dto->volume
            );

            $this->obraPublicacaoRepository->update($existingPublicacao);
        } else {
            $obraPublicacao = new ObraPublicacao(
                $obraId,
                $dto->editora,
                $dto->num_edicao,
                $dto->ano,
                $dto->produtora,
                $dto->local_publicacao,
                $dto->volume
            );

            $this->obraPublicacaoRepository->save($obraPublicacao);
        }
    }

    private function getDocumentosExistentes(int $obraId): array
    {
        $obraDocumentos = ObraDocumentoModel::where('obra_id', $obraId)
            ->with(['documento', 'tipoDocumento'])
            ->orderBy('ordem')
            ->get();

        $documentos = [];
        foreach ($obraDocumentos as $obraDocumento) {
            $documento = $obraDocumento->documento;
            if ($documento) {
                $documentos[] = [
                    'id' => $documento->id,
                    'nome' => $documento->nome,
                    'caminho_arquivo' => $documento->caminho_arquivo,
                    'tamanho' => $documento->tamanho,
                    'extensao' => $documento->extensao,
                    'tipo_documento_id' => $obraDocumento->tipo_documento_id,
                    'tipo_documento_nome' => $obraDocumento->tipoDocumento->nome ?? null,
                    'ordem' => $obraDocumento->ordem,
                    'created_at' => $documento->created_at,
                    'url_download' => url('storage/' . $documento->caminho_arquivo)
                ];
            }
        }

        return $documentos;
    }
}
